%{
Exports the line data of a phase diagram into a CSV-file that can be 
imported into other plotting software for creating customized plots for 
example.

It is easy to adapt the code to generate a different format of the 
CSV-file or to use a completely different output file format.

The system Fe-C-Cr is used as an example.
%}
database = "FEDEMO";
dependent_element = "Fe";
compositions_mass_pct = containers.Map({'C',  'Cr'}, ...
                                       {0.03, 10.0});
temp_ref = 1000;  % temperature for the initial equilibrium, in K
varied_element = "Cr";
varied_element_min = 0;   % in weight fraction
varied_element_max = 1.0; % in weight fraction
varied_temp_min = 500;    % in K
varied_temp_max = 3000;   % in K
csv_file_path = "Fe_C_Cr_phase_diagram.csv";

elements = string(compositions_mass_pct.keys());

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

tc_system = session.select_database_and_elements(database, ...
    [dependent_element elements]).get_system();

calculator = tc_system.with_phase_diagram_calculation();
calculator.set_condition(...
    tc_toolbox.ThermodynamicQuantity.temperature(), temp_ref);

for element = elements
    calculator.set_condition(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(element),...
        compositions_mass_pct(element) / 100);
end

result = calculator...
    .with_first_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(varied_element))...
            .set_min(varied_element_min)...
            .set_max(varied_element_max)...
    ).with_second_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(...
        tc_toolbox.ThermodynamicQuantity.temperature())...
            .set_min(varied_temp_min)...
            .set_max(varied_temp_max)...
    ).calculate();

phase_diagram = result...
    .get_values_grouped_by_quantity_of(...
        tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(varied_element),...
        tc_toolbox.ThermodynamicQuantity.temperature()...
    );

tie_lines = phase_diagram.get_tie_lines();
invariants = phase_diagram.get_invariants();
phase_boundaries = phase_diagram.get_lines();

% the CSV-file will contain all data within two columns:
%   * between each separate line (a phase boundary or invariant section)
%     there is a row containing NaN-values
%   * no details about the phase regions are stored
%   * it is assumed that this format is suitable for the importing program
fileID = fopen(csv_file_path, 'w' );
fprintf(fileID,'%10s, %10s\r\n', varied_element + ' [wt-%]', 'Temperature [K]');
% the only label is the varied element due to group by quantity
data = phase_boundaries(upper(varied_element));
fprintf(fileID,'%6.2f, %6.2f\r\n', [data.get_x(); data.get_y()]);
fprintf(fileID,'%6.2f, %6.2f\r\n', [NaN; NaN]); 
fprintf(fileID,'%6.2f, %6.2f\r\n', [invariants.get_x(); invariants.get_y()]);
fclose(fileID);
