%{
This is example shows how a Vacuum Oxygen Decarburization (VOD) refining process could be modelled. It is inspired
by this paper: R. Ding, B. Blanpain, P.T. Jones, P. Wollants: Modeling of the Vacuum Oxygen
Decarburization Refining Process, Metallurgical and Materials Transactions 31B (2000) 197 - 206

The validation data has been derived from that paper as well.

The calculation requires the database TCOX9 or higher (slight model adjustments might be required based on the version).

This model is showing several features of process simulations, such as:
* time-dependent mass transfer coefficients
* time-dependent pressure
* inclusion flotation
* cooling due to heat losses
* heat transfer between the zones
* control of degassing
%}

import tc_toolbox.process_metallurgy.base.*;
import tc_toolbox.process_metallurgy.process.*;
import tc_toolbox.*

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

database = ProcessDatabase.TCOX12;

steel_mass_transfer_coeff_blowing = 2.0e-3;  % in m/s
steel_mass_transfer_coeff_no_blowing = 6.0e-4;  % in m/s
steel_density = 7800;  % in kg/m**3

slag_mass_transfer_coeff_blowing = 4.0e-3;  % in m/s
slag_mass_transfer_coeff_no_blowing = 1.2e-3;  % in m/s
slag_density = 4500;  % in kg/m**3
heat_transfer_coefficient = 5.0e3;  % in W/(m**2 * K)
flotation_rate = 5.0;  % in %/min
heat_loss_steel_in_mw = -3.0;  % in MW

area = 10.0;  % in m**2

blowing_pressure_in_bar = 0.15;
time_blowing_in_min = 45;
oxygen_flow = 33.333;  % in Nm**3/min
degassing_pressure_in_pa = 200;
degassing_time_in_min = 10;
reduction_time_in_min = 40;

reference_file_path = "ding_et_al_2000_measured_data_fig_3.json";

time_end_in_min = time_blowing_in_min + degassing_time_in_min + reduction_time_in_min;

steel = SingleTimeAddition(containers.Map(["Fe", "C", "Si", "Mn", "Cr", "Ni"], { NaN, 0.48, 0.2, 0.55, 16.45, 10.49}),...
                               121e3, 1549 + 273.15);
slag = SingleTimeAddition(containers.Map(["CaO", "SiO2", "MgO"], { NaN, 12, 28}), 1200, 1549 + 273.15);
oxygen = ContinuousGasAddition(containers.Map({'O2'}, {100}), oxygen_flow / 60);

% deduced the compositions by analysis of the data in Ding et al. (2000)
metallic_reducer = SingleTimeAddition(containers.Map(["Fe", "Si", "Mn", "Al", "Cr"], { 0.09, 0.32, 0.35, 0.12, 0.12}),...
                                      937.5, 293.15, CompositionUnit.MASS_FRACTION);
slag_reducer = SingleTimeAddition(containers.Map(["CaO", "MgO"], { 80, 20}), 2000 / 4);

calc = session.with_metallurgy().with_adiabatic_process_calculation(database);

steel_zone = MetalBulkZone(steel_density);
slag_zone = SlagBulkZone(slag_density);

steel_mass_transfer_coeff = MassTransferCoefficients();
steel_mass_transfer_coeff.add(steel_mass_transfer_coeff_blowing, 0.0);
steel_mass_transfer_coeff.add(steel_mass_transfer_coeff_no_blowing, time_blowing_in_min * 60);

slag_mass_transfer_coeff = MassTransferCoefficients();
slag_mass_transfer_coeff.add(slag_mass_transfer_coeff_blowing, 0.0);
slag_mass_transfer_coeff.add(slag_mass_transfer_coeff_no_blowing, time_blowing_in_min * 60);

reaction_zone = ReactionZone(area, steel_zone, steel_mass_transfer_coeff, slag_zone, slag_mass_transfer_coeff);
(reaction_zone...
 .add_heat_transfer(heat_transfer_coefficient)...
 .add_transfer_of_phase_group(TransferOfPhaseGroup(PhaseGroup.ALL_SLAG, steel_zone)...
                              .add(flotation_rate / 60)));

(calc...
 .set_end_time(60 * time_end_in_min)...
 .with_reaction_zone(reaction_zone));

% initial zones
steel_zone.add_addition(steel, 0);
slag_zone.add_addition(slag, 0);

% degassing happens only at the interface
steel_zone.disable_degassing();
slag_zone.disable_degassing();

% heat loss
steel_zone.add_power(1e6 * heat_loss_steel_in_mw);

% pressure control
calc.set_pressure_in_time_period(1.0e5 * blowing_pressure_in_bar, 0.0, time_blowing_in_min * 60);
calc.set_pressure_in_time_period(degassing_pressure_in_pa, time_blowing_in_min * 60);

% oxygen blowing
steel_zone.add_continuous_addition(oxygen, 0, time_blowing_in_min * 60);

% reduction step
steel_zone.add_addition(metallic_reducer, (time_blowing_in_min + degassing_time_in_min) * 60);
slag_zone.add_addition(slag_reducer, (time_blowing_in_min + degassing_time_in_min) * 60);
steel_zone.add_addition(metallic_reducer, (time_blowing_in_min + degassing_time_in_min + 1) * 60);
slag_zone.add_addition(slag_reducer, (time_blowing_in_min + degassing_time_in_min + 1) * 60);
steel_zone.add_addition(metallic_reducer, (time_blowing_in_min + degassing_time_in_min + 2) * 60);
slag_zone.add_addition(slag_reducer, (time_blowing_in_min + degassing_time_in_min + 2) * 60);
steel_zone.add_addition(metallic_reducer, (time_blowing_in_min + degassing_time_in_min + 3) * 60);
slag_zone.add_addition(slag_reducer, (time_blowing_in_min + degassing_time_in_min + 3) * 60);

result = calc.calculate();
times = result.get_time_points();

% plot the results

json = fileread('ding_et_al_2000_measured_data_fig_3.json');
reference_data = jsondecode(json);

figure();
axis padded
hold on
steel_composition_ref = reference_data.data.steel.data;
for component = ["Cr", "Mn", "Si", "C", "Ni"]
    comp = result.get_composition("metal");
    line = plot(times / 60, comp(component), 'DisplayName', component, 'LineStyle', "-");
    if ismember(component, fieldnames(steel_composition_ref))
        color = get(line, 'Color');
        plot(steel_composition_ref.(component).x, steel_composition_ref.(component).y, "--", 'Color', color, 'HandleVisibility','off');
    end
end
title("Ref. (dashed): Ding et al., Met Mater Trans 31B (2000) 197");
xlabel("time / min");
ylabel("Composition of the steel melt / wt-%");
legend()
hold off

figure();
axis padded
hold on
slag_composition_ref = reference_data.data.slag.data;
slag_composition = result.get_composition_of_phase_group("slag", PhaseGroup.ALL_SLAG);
for component = ["Al2O3", "CaO", "Cr2O3", "MgO", "MnO", "SiO2"]
    if component == "Cr2O3"
        line = plot(times / 60, slag_composition("Cr2O3") + slag_composition("CrO"), 'DisplayName', "Cr2O3 + CrO", 'LineStyle', "-");
    else
         line = plot(times / 60, slag_composition(component), 'DisplayName', component, 'LineStyle', "-");
    end
    if ismember(component, fieldnames(slag_composition_ref))
        color = get(line, 'Color');
        plot(slag_composition_ref.(component).x, slag_composition_ref.(component).y, "--", 'Color', color, 'HandleVisibility','off');
    end
end
xlabel("time / min")
ylabel("Composition of the slag / wt-%")
legend()
title("Ref. (dashed): Ding et al., Met Mater Trans 31B (2000) 197")
hold off

figure()
axis padded
hold on
plot(times / 60, result.get_pressure("metal"))
xlabel("time / min")
ylabel("pressure / Pa")
hold off

figure()
axis padded
hold on
plot(times / 60, result.get_temperature("metal") - 273.15, 'DisplayName', "steel")
plot(times / 60, result.get_temperature("slag") - 273.15, 'DisplayName', "slag")
plot(reference_data.data.temperature.data.x,...
         reference_data.data.temperature.data.y,...
         "--", 'DisplayName', "reference")
xlabel("time / min")
ylabel("temperature C")
legend()
title("Ref. (dashed): Ding et al., Met Mater Trans 31B (2000) 197")
hold off

figure()
axis padded
hold on
slag_masses_in_kg = result.get_amount("slag");
oxide_pct_in_slag = result.get_composition_of_phase_group("slag", PhaseGroup.ALL_SLAG);
cr_oxides_in_slag_masses_in_kg = slag_masses_in_kg .* transpose((oxide_pct_in_slag("Cr2O3") + oxide_pct_in_slag("CrO")) / 100);
line = plot(times / 60, slag_masses_in_kg, 'DisplayName', "total");
slag_mass_ref = reference_data.data.total_slag_weight.data;
color = get(line, 'Color');
plot(slag_mass_ref.x, slag_mass_ref.y, "--", 'Color', color, 'HandleVisibility','off')

line = plot(times / 60, cr_oxides_in_slag_masses_in_kg, 'DisplayName', "Cr-oxides");
cr_oxide_slag_mass_ref = reference_data.data.cr2o3_slag_weight.data;
color = get(line, 'Color');
plot(cr_oxide_slag_mass_ref.x, cr_oxide_slag_mass_ref.y, "--", 'Color', color, 'HandleVisibility','off')

xlabel("time / min")
ylabel("slag mass / kg")
legend()
title("Ref. (dashed): Ding et al., Met Mater Trans 31B (2000) 197")
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("Exhaust gas composition / wt-%")
map = result.get_exhaust_gas().get_composition();
for item = keys(map)
    component = item{1};
    content = map(component);
    plot(times / 60, content, 'DisplayName', component)
end
legend()
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("Accumulated exhaust gas amount / kg")
map = result.get_exhaust_gas().get_amount_of_components();
for item = keys(map)
    component = item{1};
    accumulated_amount = map(component);
    plot(times / 60, accumulated_amount, 'DisplayName', component)
end
legend()
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("Phase fraction in steel zone / -")
map = result.get_fraction_of_phases("metal");
for item = keys(map)
    phase = item{1};
    fraction = map(phase);
    plot(times / 60, fraction, 'DisplayName', phase)
end
legend()
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("Phase fraction in slag zone / -")
map = result.get_fraction_of_phases("slag");
for item = keys(map)
    phase = item{1};
    fraction = map(phase);
    plot(times / 60, fraction, 'DisplayName', phase)
end
legend()
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("Phase fraction in reaction zone / -")
map = result.get_fraction_of_phases("slag_metal");
for item = keys(map)
    phase = item{1};
    fraction = map(phase);
    plot(times / 60, fraction, 'DisplayName', phase)
end
legend()
hold off

figure()
subplot(3,1,1)
axis padded
hold on
time_points_minus_one = times(1: length(times)-1) / 60;
plot(time_points_minus_one, diff(times), 'bo-');
xlabel("time / min")
ylabel("time step / s", 'Color', "b")

subplot(3,1,2)
plot(time_points_minus_one, diff(result.get_num_of_performed_steps()), "ro-")
xlabel("time / min")
ylabel("Number of calculations per time-step", 'Color', "r")

subplot(3,1,3)
plot(times / 60, result.get_num_of_performed_steps(), "go-")
xlabel("time / min")
ylabel("Number of performed time-step / s", 'Color', "r")
hold off

figure()
axis padded
hold on
xlabel("time / min")
ylabel("total mass of element / kg")
map = result.get_amount_of_elements();
for item = keys(map)
    element = item{1};
    amount = map(element);
    plot(times / 60, amount, 'DisplayName', element)
end
legend()
hold off



