%{
This example shows how to create isothermal equilibrium calculations with
varied composition of two steel components.

This example requires the database TCOX9 or higher.
%}

import tc_toolbox.process_metallurgy.base.*;
import tc_toolbox.process_metallurgy.equilibrium.*;
import tc_toolbox.*

session = tc_toolbox.TCToolbox();

[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

database = ProcessDatabase.TCOX12;

mn_conc_range_in_wt_pct = linspace(0.1, 2.3, 11);
cr_conc_range_in_wt_pct = linspace(10.0, 21.0, 11);
material_temp_in_c = 1600;
unit = CompositionUnit.MASS_PERCENT;

steel_composition = containers.Map(["Fe", "C", "Mn", "S", "Cr", "Ni", "O"], {NaN, 0.03, 1.0, 0.02, 17.5, 7.9, 0.01});
steel_mass_in_t = 95;

% note that this slag composition is not summing to 100%, such data is frequent in industrial context
% you can set the flag `do_scale` on the `EquilibriumAddition` to scale the composition automatically to 100%
slag_composition = containers.Map(["CaO", "SiO2", "MgO", "MnO", "FeO", "Cr2O3", "Al2O3"], {40, 10, 8, 4, 3, 25, 4});
slag_mass_in_t = 7;

steel = EquilibriumAddition(steel_composition, 1e3 * steel_mass_in_t, material_temp_in_c + 273.15, unit);
slag = EquilibriumAddition(slag_composition, 1e3 * slag_mass_in_t, material_temp_in_c + 273.15, unit, true);

calc = session.with_metallurgy().with_isothermal_equilibrium_calculation(database);
(calc...
 .add_addition(steel)...
 .add_addition(slag)...
 .set_temperature(material_temp_in_c + 273.15));

% ferro_silicon
calc.add_addition(EquilibriumAddition(containers.Map(["Fe", "Si"], {20, NaN}), 100, material_temp_in_c + 273.15, unit));

% aluminium
calc.add_addition(EquilibriumAddition(containers.Map({'Al'}, {100}), 500, material_temp_in_c + 273.15, unit));

% lime
calc.add_addition(EquilibriumAddition(containers.Map({'CaO'}, {100}), 500, material_temp_in_c + 273.15, unit));

% silica
calc.add_addition(EquilibriumAddition(containers.Map({'SiO2'}, {100}), 20, material_temp_in_c + 273.15, unit));


cr2o3_content_in_slag = zeros(length(mn_conc_range_in_wt_pct), length(cr_conc_range_in_wt_pct));
for mn_index = 1:length(mn_conc_range_in_wt_pct)
    mn_conc = mn_conc_range_in_wt_pct(mn_index);
    calc.update_addition(steel.set_component_composition('Mn', mn_conc));
    for cr_index = 1:length(cr_conc_range_in_wt_pct)
        cr_conc = cr_conc_range_in_wt_pct(cr_index);
        disp("Calculating for Mn=" + mn_conc + " wt-%, Cr=" + cr_conc + " wt-%");
        calc.update_addition(steel.set_component_composition('Cr', cr_conc));
        r = calc.calculate();
        comp = r.get_composition_of_phase_group(PhaseGroup.ALL_SLAG);
        cr2o3_content_in_slag(mn_index, cr_index) = comp('Cr2O3');
    end
end

[X,Y] = meshgrid(cr_conc_range_in_wt_pct, mn_conc_range_in_wt_pct);
[C,h] = contour(X,Y, cr2o3_content_in_slag);
clabel(C,h)
title("Cr2O3-content in slag / wt-%");
xlabel('Mn-content / wt-%');
ylabel('slag property');

