%{
This example shows how to create a property (step) diagram using TC-Toolbox.
The alloy system Fe-Ni is used as an example.

The result is then saved in different file formats:
    * JSON
    * XML
    * CSV
    * TXT
    * HDF5

%}

database = "FEDEMO";
dependent_element = "Fe";
composition = containers.Map(["Ni", "C"], [10.0, 1.0]);  % in wt-%
elements = [dependent_element string(composition.keys())];

session = tc_toolbox.TCToolbox();
[filepath,name,ext] = fileparts(mfilename('fullpath'));
session.set_cache_folder( name + "_cache");

system = session.select_database_and_elements(database, elements).get_system();

property_diagram_calculation = system.with_property_diagram_calculation();
property_diagram_calculation.with_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis(tc_toolbox.ThermodynamicQuantity.temperature())...
                       .set_min(500)...
                       .set_max(3000)...
                       );

for element_key = composition.keys()
    element = element_key{1};
    property_diagram_calculation.set_condition(tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(element), composition(element)/100);
end


property_diagram = property_diagram_calculation.calculate();
groups = property_diagram.get_values_grouped_by_quantity_of(tc_toolbox.ThermodynamicQuantity.temperature(),...
                                                       tc_toolbox.ThermodynamicQuantity.volume_fraction_of_a_phase('*'));

% to `*.json`
keySet = {'Title','x-axis','y-axis', 'lines'};
diagram_title = 'Property diagram Fe-Ni';
xaxis = 'T [K]';
yaxis = 'Volume fraction of phases';
lines = containers.Map();
for group = groups.values()
    label = group{1}.get_label();
    line = containers.Map({'label', 'x', 'y'}, {label, group{1}.get_x(), group{1}.get_y()},'UniformValues', false);
    lines(label) = line;
end
valueSet = {diagram_title, xaxis, yaxis, lines};
M = containers.Map(keySet,valueSet,'UniformValues',false);
fid = fopen('property_diagram.json','w');
fprintf(fid, jsonencode(M));
fclose(fid);


% to `*.txt`
fid = fopen('property_diagram.txt','w');
for group = groups.values()
    fprintf(fid, group{1}.get_label() + "\n");
    label = group{1}.get_label();
    x = group{1}.get_x();
    y = group{1}.get_y();
    for i = 1:length(x)
        if isnan(x(i)) || isnan(y(i))
            fprintf(fid, "NaN, NaN\n");
        else
            fprintf(fid, string(x(i)) + ", " + string(y(i)) + "\n");
        end
    end
    fprintf(fid,"\n");
end
fclose(fid);


% to `*.xml`
docNode = com.mathworks.xml.XMLUtils.createDocument('Property_diagram');
doc_element = docNode.getDocumentElement;

for group = groups.values()
    label = group{1}.get_label();
    x = group{1}.get_x();
    y = group{1}.get_y();

    group_xml = docNode.createElement(strrep(label,'#','_compset'));
    doc_element.appendChild(group_xml);
    for i = 1:length(x)
        curr_node = docNode.createElement('coordinate');
        if isnan(x(i)) || isnan(y(i))
            curr_node.setAttribute('x', 'NaN');
            curr_node.setAttribute('y', 'NaN');
        else
            curr_node.setAttribute('x', num2str(x(i)));
            curr_node.setAttribute('y', num2str(y(i)));
        end
        group_xml.appendChild(curr_node);
    end
end
xmlwrite('property_diagram.xml',docNode);


% to `*.HDF5`
if exist('property_diagram.f5', 'file')==2
  delete('property_diagram.f5');
end

h5create('property_diagram.f5', '/system', size(length(composition)));
h5writeatt('property_diagram.f5','/system','database', database);
h5writeatt('property_diagram.f5','/system','dependentElement', dependent_element);
h5writeatt('property_diagram.f5','/system','elements', string(composition.keys()));
h5writeatt('property_diagram.f5','/system','compositions', cell2mat(composition.values()));

for group = groups.values()
    label = group{1}.get_label();
    x = group{1}.get_x();
    y = group{1}.get_y();

    h5create('property_diagram.f5', '/results/' + label + '/x', length(x))
    h5write('property_diagram.f5', '/results/' + label + '/x', x)
    h5writeatt('property_diagram.f5','/results/' + label + '/x','unit', 'Kelvin');

    h5create('property_diagram.f5', '/results/' + label + '/y', length(y))
    h5write('property_diagram.f5', '/results/' + label + '/y', y)
    h5writeatt('property_diagram.f5','/results/' + label + '/y','unit', 'volume fraction');

end


% to `*.CSV`
% note: due to the differing temperature axis range for each phase we choose here only one phase
group_temp = groups.values();
group = group_temp(1);

label = group{1}.get_label();
x = group{1}.get_x();
y = group{1}.get_y();

fileID = fopen("property_diagram.csv", 'w' );
fprintf(fileID,'%10s, %10s\r\n', 'Temperature [K]', label);
fprintf(fileID,'%6.2f, %6.2f\r\n', [x y]');
fclose(fileID);