%{
Deriving Diffusion Coefficients from Experiment

This example demonstrates the direct optimization of mobility database parameters by fitting 
to experimentally measured composition profiles. A manual determination of the interdiffusion
coefficients (for example using the Boltzmann-Matano method) is not required. Additionally, 
no derivatives of the experimental curve are required as this is difficult for scattered data.

This example has set the maximum number of iterations to 10 in order to keep runtime low. 
The results are updated continually in the Results window. The plot example shown is the final iteration.

The example uses data from:
* Rettig, Ralf, Susanne Steuer, and Robert F. Singer. 2011. “Diffusion of Germanium in Binary 
  and Multicomponent Nickel Alloys.” Journal of Phase Equilibria and Diffusion 32 (3): 198–205. 
  doi:10.1007/s11669-011-9853-6.
* Liu, Y.Q., D.J. Ma, and Y. Du. 2010. “Thermodynamic Modeling of the Germanium–nickel System.”
  Journal of Alloys and Compounds 491 (1–2): 63–71. doi:10.1016/j.jallcom.2009.11.036.
%}

jsonData = jsondecode(fileread("data_for_D_06/NiGe_1150C_10h.json"));
d = jsonData.data;
distance = [d.x]./1e6;
composition = [d.c];
width = max(distance);

param_NiNiGe = "MQ(FCC_A1&Ni,Ni,Ge:VA;0)";
param_GeGeNi = "MQ(FCC_A1&Ge,Ge,Ni:VA;0)";

data_directory_thermo = "data_for_D_06/NiGe_thermo.tdb";
data_directory_mob = "data_for_D_06/NiGe_mob.tdb";
temperature = 1150 + 273.15;  % in K
simulation_time = 10 * 3600;  % in s
max_conc = 13.5;  % maximum Ge-concentration in at-%

session = tc_toolbox.TCToolbox();

tc_system = session...
    .select_user_database_and_elements(data_directory_thermo, ["Ni", "Ge"])...
    .without_default_phases().select_phase("FCC_A1")...
    .select_user_database_and_elements(data_directory_mob, ["Ni", "Ge"])...
    .without_default_phases().select_phase("FCC_A1")...
    .get_system();

diffusion_calc = tc_system.with_isothermal_diffusion_calculation();
equilibrium_calc = tc_system.with_single_equilibrium_calculation();

x0 = [1,1, 5e-4]; % initial guess for param_NiNiGe, param_GeGeNi, coordinate of concentration step

opt_results = findzero_w_modelparams(composition, distance, equilibrium_calc, diffusion_calc, x0);

function y = findzero_w_modelparams(y_exp, x_exp, equilibrium_calc, diffusion_calc, x0)
    options = optimset("MaxIter",10);
    y = fminsearch(@poly, x0, options);
   
    function y = poly(x)
        
        width = 0.0012;
        temperature = 1150 + 273.15;  % in K
        simulation_time = 10 * 360;  % in s
        max_conc = 13.5;  % maximum Ge-concentration in at-%
        
        diffusion_calc.with_system_modifications(...
            tc_toolbox.SystemModifications().set(tc_toolbox.PhaseParameter("MQ(FCC_A1&Ni,Ni,Ge:VA;0)")...
            .set_expression_with_upper_limit(num2str(x(1)))));
        
        equilibrium_calc.with_system_modifications(...
            tc_toolbox.SystemModifications().set(tc_toolbox.PhaseParameter("MQ(FCC_A1&Ni,Ni,Ge:VA;0)")...
            .set_expression_with_upper_limit(num2str(x(1)))));
        
        diffusion_calc.with_system_modifications(...
            tc_toolbox.SystemModifications().set(tc_toolbox.PhaseParameter("MQ(FCC_A1&Ge,Ge,Ni:VA;0)")...
            .set_expression_with_upper_limit(num2str(x(2)))));
        
        equilibrium_calc.with_system_modifications(...
            tc_toolbox.SystemModifications().set(tc_toolbox.PhaseParameter("MQ(FCC_A1&Ge,Ge,Ni:VA;0)")... % set the parameter name(s)
            .set_expression_with_upper_limit(num2str(x(2)))));
        
        diffusion_result = (diffusion_calc...
            .remove_all_regions()...
            .add_region(tc_toolbox.diffusion.Region("FCC")...
            .add_phase("FCC_A1")...
            .set_width(width)...
            .with_grid(tc_toolbox.diffusion.CalculatedGrid.automatic())...
            .with_composition_profile(tc_toolbox.diffusion.CompositionProfile(tc_toolbox.diffusion.Unit.MOLE_PERCENT)...
            .add("Ge", tc_toolbox.diffusion.ElementProfile.step(1e-6, max_conc, x(3)))))...
            .set_temperature(temperature)...
            .set_simulation_time(simulation_time)...
            .calculate());
        
        [distance, mf_ge] = diffusion_result.get_mole_fraction_of_component_at_time("Ge", simulation_time);
        
        interpol_res = interp1(distance, mf_ge*100, x_exp);
        y = sum((y_exp - interpol_res).^2);
        disp(["Intermediate fitting result: ", x(1), x(2), x(3)]);
        subplot(2,1,1)
        title("Ge profile in Ni-Ge");
        plot(distance, mf_ge*100);
        hold on;
        plot(x_exp, y_exp, "o");
        legend("Calculated", "Experiment")
        xlabel("Distance")
        ylabel("at% Ge")
        hold off
        pause(1.0)
        % calculate how the diffusivity varies
        subplot(2,1,2)
        [ge_linspace, D_ge] = calculate_diffusion_coefficient("Ge", 0, max_conc, 100, temperature, "Ni", equilibrium_calc);
        plot(ge_linspace, D_ge)
        xlabel("at% Ge")
        ylabel("D_{GeGeNi}")
        legend("Diffusivity")
    end
end

function [c_linspace, diffusion_coeff] = calculate_diffusion_coefficient(element, c_min, c_max, num_points, temperature, dependent_element, equilibrium_calc)

    %Calculates the diffusivities of the element for the specified composition range
    (equilibrium_calc...
     .disable_global_minimization()...
     .set_condition(tc_toolbox.ThermodynamicQuantity.temperature(), temperature));
    c_linspace = linspace(c_min, c_max, num_points);

    diffusion_coeff = zeros([1, length(c_linspace)]);
    for i = 1:length(c_linspace)
        equilibrium_calc.set_condition(tc_toolbox.ThermodynamicQuantity.mole_fraction_of_a_component(element), c_linspace(i) / 100);
        result = equilibrium_calc.calculate();
        diffusion_coeff(i) = result.get_value_of(sprintf("DC(FCC,%s,%s,%s)",element,element,dependent_element));
    end
end
