%{
Diffusion in a Single Region with Multiple Phases

This example simulates diffusion in a single region having multiple phases. Initially a 
step concentration profile is defined within the single region. Due to the multiple phases,
the default `AutomaticSolver` uses the 'homogenization mode'. 

The example uses a minimum number of required settings. Default values are used for any 
unspecified settings.
%}

session = tc_toolbox.TCToolbox();
[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
            .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "Cr", "Ni"])...
            .without_default_phases()...
            .select_phase("FCC_A1")...
            .select_phase("BCC_A2")...
            .get_system();

calculator = tc_system...
                .with_isothermal_diffusion_calculation()...
                .set_temperature(1100 + 273.15)...
                .set_simulation_time(100.0 * 3600)...
                .add_region(tc_toolbox.diffusion.Region("Diffcouple")...
                    .set_width(0.003)...
                        .with_grid(tc_toolbox.diffusion.CalculatedGrid.automatic().with_fine_grid())...
                    .with_composition_profile(tc_toolbox.diffusion.CompositionProfile(tc_toolbox.diffusion.Unit.MOLE_PERCENT)...
                        .add("Cr", tc_toolbox.diffusion.ElementProfile.step(25.7, 42.3, 0.0015))...
                        .add("Ni", tc_toolbox.diffusion.ElementProfile.step(6.47, 27.5, 0.0015)))...
                    .add_phase("FCC_A1")...
                    .add_phase("BCC_A2"))...
                .with_solver(tc_toolbox.diffusion.Solver.homogenization()...
                    .with_function(tc_toolbox.diffusion.HomogenizationFunctions.general_lower_hashin_shtrikman()));

result = calculator.calculate();
[distance, mole_frac_fcc] = result.get_mass_fraction_of_phase_at_time("FCC", tc_toolbox.diffusion.SimulationTime.LAST);

plot(distance, mole_frac_fcc);
title("Multiphase diffusion couple");
xlabel("'Distance [m");
ylabel("Phase fraction of FCC");
