"""
This example shows how to invoke dynamic arguments (equivalent to clicking the add (+) button in the GUI)
for a property model field that supports this. One common example is adding multiple precipitates to a
yield strength calculation.
"""
from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

db = "TCAL8"
dep_el = "AL"
els_and_comps = {"CU": 4.0, "MG": 0.25}  # Mass percent
# Create a list of radii. Here a logarithmic spacing is chosen to resolve the quickly changing strength contribution
# at small radii.
radii = np.logspace(-10, -8, 25)
equilibrium_temperature = 298.15
precipitate_phase_1 = "AL2CU_C16"
precipitate_phase_2 = "T_PHASE"
results = []

with TCPython() as session:

    system = (session
              .disable_caching()
              .select_database_and_elements(db, [dep_el] + list(els_and_comps.keys()))
              .get_system())

    c = (system.with_property_model_calculation("Yield Strength", debug_model=False)
         .set_temperature(equilibrium_temperature)
         .set_composition_unit(CompositionUnit.MASS_PERCENT)
         .set_argument("Matrix", "FCC_A1")
         .set_argument("Select mode", "Advanced")
         .set_argument("precip_str_selection", True)
         .set_argument("Precipitate-1", precipitate_phase_1)
         .set_argument("sol_str_selection", True)
         .set_argument("grain_str_selection", True)
         .set_argument("Anti_phase boundary energy Seidman-1", 0.46)  # Default value = 0.46
         .set_argument("prefactor 'm'-1", 0.85)  # Default value = 0.85
         .set_argument("seidman alpha-1", 2.6)  # Default value = 2.6
         .set_argument("Precipitation model list-1", "Seidman model (Al-base)")
         )

    for element, composition in els_and_comps.items():
        c.set_composition(element, composition)

    # Creates "Precipitate-2", equivalent to pressing '+' next to a precipitate in the GUI version
    print(c.get_dynamic_arguments())
    c.invoke_dynamic_argument("Precipitate-1")
    print(c.get_arguments())

    (c.set_argument("Select mode", "Advanced")
     .set_argument("precip_str_selection", True)
     .set_argument("Precipitate-2", precipitate_phase_2)
     .set_argument("sol_str_selection", True)
     .set_argument("grain_str_selection", True)
     .set_argument("Anti_phase boundary energy Seidman-2", 0.26)  # Default value = 0.46
     .set_argument("prefactor 'm'-2", 0.85)  # Default value = 0.85
     .set_argument("seidman alpha-2", 2.6)  # Default value = 2.6
     .set_argument("Precipitation model list-2", "Seidman model (Al-base)")
     )
    for radius in radii:
        c.set_argument('Precipitate radius seid-1', radius)
        c.set_argument('Precipitate radius seid-2', radius)
        r = c.calculate()
        precip_results = r.get_value_of("Precipitation strengthening per phase")
        results.append(precip_results)

# For easier plotting when calculations are finished
result_for_plotting = {}
for key in results[0].keys():
    result_for_plotting[key] = []

# Add the results for each precipitate phase to it´s own list
for result in results:
    for key, value in result.items():
        result_for_plotting[key].append(value)

# Visualize the result
for key, value in result_for_plotting.items():
    plt.plot(radii, value, label=key)
plt.xlabel("Particle radius (m)")
plt.ylabel("Precipitation strengthening per phase")
plt.legend()
plt.show()
