from tc_python import *
import matplotlib.pyplot as plt
import numpy as np
from io import StringIO

"""
The isothermal coarsening kinetics of γ' precipitates in a Ni-Al-Cr alloy is modeled. The
experimental data is from [2008Sud], which investigates isothermal coarsening kinetics at a
temperature of 873 K for up to 264 h.
It is assumed that there are no precipitates in solution at the beginning of the isothermal heat
treatment.

Reference:
[2008Sud] C. K. Sudbrack, T. D. Ziebell, R. D. Noebe, D. N. Seidman, Effects of a tungsten addition on the
            morphological evolution, spatial correlations and temporal evolution of a model Ni–Al–Cr superalloy. 
            Acta Mater. 56, 448–463 (2008).
"""
def unit_conversion(data, unit_conversion):
    converted_result = np.zeros(len(data), dtype=float)
    for i in range(len(data)):
        converted_result[i] = data[i]*unit_conversion
    return converted_result


temperature = 800.
simulation_time = 300.*3600.

precipitate_phase = "FCC_L12#2"
matrix_phase = "DIS_FCC_A1"


histogram_list = []
data_file = StringIO("""0.001,0
3.121,4.6269421315804e+19
6.241,1.32840759216992e+21
9.361,2.32001995888184e+21
12.481,2.56545704549115e+21
15.601,3.61512091136382e+21
18.721,4.71467900421334e+21
21.841,4.87718550906963e+21
24.961,4.2101821037073e+21
28.081,2.58118459354551e+21
31.201,2.06352509363756e+21
34.321,1.28062186330576e+21
37.441,5.80447880047813e+20
40.561,8.78689812785799e+19
43.681,7.60869034336303e+19
46.801,0""")
histogram_4h = np.genfromtxt(data_file, delimiter=',')
histogram_list.append(histogram_4h)

data_file = StringIO("""0,0
5.595,0
11.19,6.01159483152872e+19
16.785,4.22754518341054e+20
22.38,6.46595922563864e+20
27.975,9.4497947291663e+20
33.57,1.27421374042102e+21
39.165,1.23073641655158e+21
44.76,1.16926673220035e+21
50.355,8.0448692818188e+20
55.95,4.93686034647366e+20
61.545,1.44306843388102e+20
67.14,4.17156907034556e+19
72.735,5.92242812296639e+18
78.33,0""")
histogram_16h = np.genfromtxt(data_file, delimiter=',')
histogram_list.append(histogram_16h)

data_file = StringIO("""0,0
8.85,4.36692196908933e+18
17.7,3.00833537869728e+19
26.55,9.02497924137562e+19
35.4,1.45078974604564e+20
44.25,2.47459090879837e+20
53.1,3.27034370421642e+20
61.95,3.18785760953697e+20
70.8,3.10537151485752e+20
79.65,1.82440576497879e+20
88.5,1.04806288626441e+20
97.35,2.42607015124663e+19
106.2,1.1159908710482e+19
115.05,4.36692196908933e+18
123.9,0""")
histogram_64h = np.genfromtxt(data_file, delimiter=',')
histogram_list.append(histogram_64h)

data_file = StringIO("""0,0
12.15,1.28583445366752e+18
24.3,2.49622840655675e+19
36.45,4.01723019258806e+19
48.6,6.38480718936294e+19
60.75,8.54554233078145e+19
72.9,1.02169441394316e+20
85.05,1.0100798182031e+20
97.2,1.14524221895304e+20
109.35,8.98429968263993e+19
121.5,4.55951303276337e+19
133.65,1.88456958264423e+19
145.8,2.63296235370544e+18
157.95,0
170.1,0
""")
histogram_264h = np.genfromtxt(data_file, delimiter=',')
histogram_list.append(histogram_264h)

RbarData = np.array([
    [4, 20.8],
    [16, 37.3],
    [64, 59],
    [264, 81],
])

with TCPython():
    system = (SetUp()
              .set_cache_folder("cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("NIDEMO", "MNIDEMO", ["Ni", "Al", "Cr"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    precipitate = (PrecipitatePhase(precipitate_phase)
                   .set_interfacial_energy(0.01)
                   )

    matrix = (MatrixPhase(matrix_phase)
                       .set_mobility_adjustment(element="all", prefactor=0.5)
                       .add_precipitate_phase(precipitate))

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MOLE_PERCENT)
                    .set_composition("Al", 10.0)
                    .set_composition("Cr", 8.5)
                    .set_temperature(temperature + 273.15)
                    .set_simulation_time(simulation_time)
                    .with_matrix_phase(matrix))

    result = prisma_setup.calculate()

    time_sec, mean_radius_meters = result.get_mean_radius_2d_of(precipitate_phase)

    time_hours = unit_conversion(time_sec, 1./ 3600.)
    mean_radius_nm = unit_conversion(mean_radius_meters, 1e9)

    fig0, ax0 = plt.subplots()
    ax0.scatter(RbarData[:, 0], RbarData[:, 1], marker='x', s=100, label="2D data", color='black')
    ax0.plot(time_hours, mean_radius_nm, color='red', label='2D model')
    ax0.set_xlabel('Time [h]')
    ax0.set_ylabel('Mean radius [nm]')
    plt.legend()
    plt.show()

    i = 0
    for histogram in histogram_list:
        fig1, ax1 = plt.subplots()
        time = RbarData[i, 0] * 3600
        title = str(int(RbarData[i, 0]))+"h"
        i += 1
        radius_m, size_distribution_m = result.get_size_distribution_2d_for_radius_of(precipitate_phase, time)

        radius_nm = unit_conversion(radius_m, 1e9)

        bins_width = np.diff(histogram[:, 0])
        bins_width = np.append(bins_width, bins_width[-1])

        ax1.bar(histogram[:, 0], histogram[:, 1], align='edge', width=bins_width, color='white', edgecolor='black',
                label='2D data')
        ax1.plot(radius_nm,size_distribution_m, color='red', label='2D model')
        ax1.set_xlabel('Radius [m]')
        ax1.set_ylabel("Size distribution [m$^{-3}$]")
        ax1.set_title(title)
        plt.legend()
        plt.show()



