from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
In this example, the effect of the initial particle size distribution (PSD) on overall precipitation
kinetics is demonstrated. Two Precipitation Calculators are utilized to simulate and compare carbide 
precipitations in a Fe-0.1C-12Cr alloy within a ferritic BCC_A2 matrix. Competitive precipitations 
involving three carbides, CEMENTITE, M23C6, and M7C3, are analyzed, and the precipitation kinetics 
are compared in terms of the presence or absence of the initial particle size distribution.
"""

matrix_phase = "BCC_A2"
cementite = "CEMENTITE_D011"
phase_m23c6 = "M23C6_D84"
phase_m7c3 = "M7C3_D101"

simulation_time = 400000
temperature = 1053

# settings for the LSW Distribution
mean_radius = 1e-8
nether = 5e-10
derived_factor = 0.96019279
samples = 1000
max_raa = 1.5


# static helper methods to create the LSW distribution
def create_size_distribution(initial_composition, volume):
    p = ParticleSizeDistribution() \
        .set_initial_composition("Cr", initial_composition) \
        .set_volume_fraction_of_phase_value(volume) \
        .set_volume_fraction_of_phase_type(VolumeFractionOfPhaseType.VOLUME_FRACTION)
    list_of_radii, list_of_number_densities = compute_lsw(mean_radius)
    # uncomment the commands below to see the LSW distribution
    # plt.plot(list_of_radii, list_of_number_densities, "r-")
    # plt.show()
    for radius, number_density in zip(list_of_radii, list_of_number_densities):
        p.add_radius_and_number_density(radius, number_density)
    return p


# static helper methods to create the LSW distribution
def compute_lsw(mean_radius):
    mean_calc = derived_factor * mean_radius
    step = (max_raa * mean_calc - nether) / samples
    list_of_radii = []
    generated_lsw = []
    for i in range(0, samples):
        boundaries = nether + i * step
        r = nether + 0.5 * ((i * step) + (i + 1) * step)
        raa = r / mean_calc
        if raa > max_raa:
            density = None
        else:
            density = raa

        if density is not None and density > 0:
            f, r = distribution_value(density)
            list_of_radii.append(boundaries)
            generated_lsw.append(f / mean_calc)

    return list_of_radii, generated_lsw


# static helper methods to create the LSW distribution
def distribution_value(r):
    if r < 1.5:
        f = 4 / 9
        f *= pow(r, 2)
        f *= pow((3 / (3 + r)), (7 / 3))
        f *= pow(1.5 / (1.5 - r), 11.0 / 3.0)
        f *= np.exp(-(1.5 / (1.5 - r)))
    else:
        f = 0.0
    return f, r


with TCPython():
    system = (SetUp()
              .set_cache_folder("cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO",
                                                                        ["Fe", "Cr", "C"])
              .select_phase(matrix_phase)
              .select_phase(cementite)
              .select_phase(phase_m23c6)
              .select_phase(phase_m7c3)
              .get_system())

    precipitate_cem = (PrecipitatePhase(cementite).set_interfacial_energy(0.167)
                       .set_nucleation_at_grain_boundaries())
    precipitate_phase_m23c6 = (PrecipitatePhase(phase_m23c6).set_interfacial_energy(0.252)
                               .set_nucleation_at_grain_boundaries())
    precipitate_phase_m7c3 = (PrecipitatePhase(phase_m7c3).set_interfacial_energy(0.282)
                              .set_nucleation_at_grain_boundaries())

    matrix = (MatrixPhase(matrix_phase))
    matrix_no_psd = (matrix.add_precipitate_phase(precipitate_cem)
                     .add_precipitate_phase(precipitate_phase_m23c6)
                     .add_precipitate_phase(precipitate_phase_m7c3))

    prisma_setup = (system.with_isothermal_precipitation_calculation()
                    .set_composition_unit(CompositionUnit.MASS_PERCENT)
                    .set_composition("Cr", 12.0)
                    .set_composition("C", 0.1)
                    .set_simulation_time(simulation_time)
                    .set_temperature(temperature)
                    )

    result_no_psd = (prisma_setup.with_matrix_phase(matrix_no_psd).calculate())

    # results for the system WITHOUT PSD included
    t_no_cem, volume_no_cem = result_no_psd.get_volume_fraction_of(cementite)
    t_no_m23, volume_no_m23 = result_no_psd.get_volume_fraction_of(phase_m23c6)
    t_no_m7, volume_no_m7 = result_no_psd.get_volume_fraction_of(phase_m7c3)

    # add the PSD to the system setup
    precipitate_cem.with_particle_size_distribution(
        create_size_distribution(initial_composition=72.0, volume=0.001))
    precipitate_phase_m23c6.with_particle_size_distribution(
        create_size_distribution(initial_composition=70.0, volume=0.0015))
    precipitate_phase_m7c3.with_particle_size_distribution(
        create_size_distribution(initial_composition=83.0, volume=0.0015))

    # results WITH PSD in the precipitates
    result_with_psd = prisma_setup.calculate()

    # results for the system WITH PSD included
    t_with_cem, volume_with_cem = result_with_psd.get_volume_fraction_of(cementite)
    t_with_m23, volume_with_m23 = result_with_psd.get_volume_fraction_of(phase_m23c6)
    t_with_m7, volume_with_m7 = result_with_psd.get_volume_fraction_of(phase_m7c3)

    # plotting for the system
    fig1, ax1 = plt.subplots()
    plt.title("Volume fraction at Grain Boundaries with PSD", fontsize=13)
    plt.xlabel("Time [s]")
    plt.ylabel("Volume fraction")
    ax1.semilogx(t_with_cem, volume_with_cem)
    ax1.semilogx(t_with_m23, volume_with_m23)
    ax1.semilogx(t_with_m7, volume_with_m7)
    plt.xlim(1e-4, 1e6)
    plt.ylim(0, 0.02)
    plt.plot(t_with_cem, volume_with_cem, "b-", t_with_m23, volume_with_m23, "r-", t_with_m7, volume_with_m7, "g-")
    plt.legend(["CEMENTITE", "c23c6", "m7c3"], loc=2)
    plt.tight_layout()
    plt.show()

    fig2, ax2 = plt.subplots()
    plt.title("Volume fraction at Grain Boundaries without PSD", fontsize=13)
    plt.xlabel("Time [s]")
    plt.ylabel("Volume fraction")
    ax2.semilogx(t_no_cem, volume_no_cem)
    ax2.semilogx(t_no_m23, volume_no_m23)
    ax2.semilogx(t_no_m7, volume_no_m7)
    plt.xlim(1e-4, 1e6)
    plt.ylim(0, 0.02)
    plt.plot(t_no_cem, volume_no_cem, "b-", t_no_m23, volume_no_m23, "r-", t_no_m7, volume_no_m7, "g-")
    plt.legend(["CEMENTITE", "c23c6", "m7c3"], loc=2)
    plt.tight_layout()
    plt.show()
