from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
In this isothermal calculation example, the simulation of Al3Sc phase precipitation in an Al-Sc binary 
alloy is carried out. Two calculations are conducted: one assumes a spherical morphology without elastic 
strain energy, while the other assumes a cuboid shape determined by the interplay between interfacial energy 
and elastic strain energy. The simulation results are then compared with experimental data collected from 
studies by Marquis and Seidman [2001Mar] and Novotny and Ardell [2001Nov].

References:
[2001Mar] E. A. Marquis, D. N. Seidman, Nanoscale structural evolution of Al3Sc precipitates in Al(Sc)
            alloys. Acta Mater. 49, 1909–1919 (2001).
[2001Nov] G. M. Novotny, A. J. Ardell, Precipitation of Al3Sc in binary Al–Sc alloys. Mater. Sci. Eng. A
            Struct. Mater. Prop. Microstruct. Process. 318, 144–154 (2001).
"""

matrix_phase = "FCC_A1"
precipitate_phase = "AL3SC"
sphere = PrecipitateMorphology.SPHERE
cuboid = PrecipitateMorphology.CUBOID
simulation_time = 1e9
calc_from_molar_volume = TransformationStrainCalculationOption.CALCULATE_FROM_MOLAR_VOLUME

# data for the mean radius vs time plot
symbol1 = np.array([
    [9.4120e+04, 5.0887],
    [1.6403e+05, 4.5168],
    [4.0705e+05, 4.7468],
    [3.4983e+05, 5.1395],
    [5.7964e+05, 5.1908],
    [4.9816e+05, 5.7902],
    [2.5839e+05, 5.8480],
    [6.7445e+05, 6.2072],
    [7.4612e+05, 6.7207],
    [1.1175e+06, 6.7207],
    [1.9475e+06, 8.6155],
    [2.2660e+06, 8.7015],
    [4.3687e+06, 1.0614e+01],
    [1.0841e+07, 1.4300e+01]
])

symbol2 = np.array([
    [1.8140e+04, 2.5131],
    [8.7021e+04, 3.7696],
    [1.7268e+05, 4.6073],
    [4.3356e+05, 6.7016],
])


# static helper method to get the results for morphologies
def get_results(results):
    time_1, mean_radius_1 = results.get_mean_radius_of(precipitate_phase)
    for time in np.array(time_1):
        radius, size_distribution = results.get_size_distribution_for_radius_of(precipitate_phase, time)

    return time_1, mean_radius_1, radius, size_distribution


with TCPython():
    system = (SetUp()
              .set_cache_folder("cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("ALDEMO", "MALDEMO", ["Al", "Sc"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    setup = (system.with_isothermal_precipitation_calculation()
             .set_composition_unit(CompositionUnit.MOLE_PERCENT)
             .set_composition("Sc", 0.18)
             .set_temperature(350 + 273.15)
             .set_simulation_time(simulation_time))

    precip = (PrecipitatePhase(precipitate_phase).set_precipitate_morphology(sphere))

    matrix = (MatrixPhase(matrix_phase).add_precipitate_phase(precip))

    result_sphere = (setup.with_matrix_phase(matrix).calculate())

    precip.set_precipitate_morphology(cuboid).set_transformation_strain_calculation_option(calc_from_molar_volume)
    matrix.with_elastic_properties_cubic(108.2, 61.3, 28.5)
    result_cuboid = (setup.calculate())

    # use a static method to get the results for both morphologies
    time_sphere, mean_r_sphere, radius_sphere, sd_sphere = get_results(result_sphere)
    time_cub, mean_r_cub, radius_cub, sd_cub = get_results(result_cuboid)

    # Only cuboid precipitates have data for the cubic distribution factor
    for y in np.array(time_cub):
        radius, cub_distri_factor = result_cuboid.get_cubic_factor_distribution_for_radius_of(precipitate_phase, y)

    time_cubic_factor, mean_cub_factor = result_cuboid.get_mean_cubic_factor_of(precipitate_phase)

    fig1, ax1 = plt.subplots()
    ax1.set_title("Mean radius of AL3SC (bulk) precipitates comparing morphology", fontsize=12)
    ax1.set_xlabel("Time [s]")
    ax1.set_ylabel("Length [nm]")
    ax1.loglog(time_cub, np.array(mean_r_cub) / 1e-9, "b-")
    ax1.loglog(time_sphere, np.array(mean_r_sphere) / 1e-9, "r-")
    ax1.legend(["Cuboid", "Sphere"], loc=1)
    x, y = symbol1.T
    plt.scatter(x, y, marker="^", c="#474747")
    x, y = symbol2.T
    plt.scatter(x, y, marker="s", c="#141414")
    plt.xlim(1, 10e9)
    plt.ylim(0.1, 1e2)

    fig2, ax2 = plt.subplots()
    ax2.set_title("Size distribution of AL3SC (bulk) precipitates comparing morphology", fontsize=12)
    ax2.set_xlabel("Length [nm]")
    ax2.set_ylabel("Size distribution [m-4]")
    ax2.semilogy(np.array(radius_cub) * 1e9, sd_cub, "b-")
    ax2.semilogy(np.array(radius_sphere) * 1e9, sd_sphere, "r-")
    ax2.set_yscale("log")
    ax2.legend(["Cuboid", "Sphere"], loc=1)
    plt.xlim(0, 160)
    plt.ylim(1, 1e30)
    plt.tight_layout()

    fig3, ax3 = plt.subplots()
    ax3_other = ax3.twinx()
    ax3.set_title("Cubic factor distribution of AL3SC(bulk) cuboids", fontsize=12)
    ax3.set_xlabel("Length [nm]")
    ax3.set_ylabel("Size distribution [m-4]")
    ax3.plot(np.array(radius_cub) * 1e9, sd_cub, "b-")
    ax3_other.plot(np.array(radius_cub) * 1e9, cub_distri_factor, "r-")
    ax3_other.set_ylabel("Cubic factor distribution of AL3SC(bulk)")
    ax3.legend(["Size Distribution of AL3SC (Bulk)"], loc=1)
    ax3_other.legend(["Cubic distribution factor of AL3SC (Bulk)"], loc=4)
    ax3.set_ylim(0, 4.5e25)
    ax3_other.set_ylim(1.0, 1.45)
    plt.xlim(0, 160)
    plt.tight_layout()

    fig4, ax4 = plt.subplots()
    ax4_other = ax4.twinx()
    ax4.set_title("Mean Cubic factor of AL3SC(bulk) cuboids", fontsize=12)
    ax4.set_xlabel("Time [s]")
    ax4.set_ylabel("Length [nm]")
    ax4.loglog(time_cub, np.array(mean_r_cub) / 1e-9, "b-")
    ax4_other.plot(time_cub, mean_cub_factor, "r-")
    ax4_other.set_ylabel("Mean cubic factor")
    ax4.legend(["Mean radius of AL3SC"], loc=1)
    ax4_other.legend(["Mean cubic factor of AL3SC(Bulk)"], loc=4)
    ax4.set_ylim(1e-1, 1e2)
    ax4_other.set_ylim(1.0, 1.4)
    plt.xlim(0, 1e9)
    plt.tight_layout()
    plt.show()

