from tc_python import *
import matplotlib.pyplot as plt
import numpy as np

"""
This example simulates the kinetics of precipitation of gamma prime (γ’) phase from gamma (γ) phase in
Ni-8Al-8Cr and Ni-10Al-10Cr at.% alloys during continuous cooling. The simulation results can be
compared with experimental results from Rojhirunsakool et al. [2013Roj].

Reference:
[2013Roj] T. Rojhirunsakool, S. Meher, J. Y. Hwang, S. Nag, J. Tiley, R. Banerjee, Influence of composition
            on monomodal versus multimodal γ′ precipitation in Ni–Al–Cr alloys. J. Mater. Sci. 48, 825–831 (2013).
"""

initial_temperature = 1150 + 273.15
simulation_time = 3300
lower_temperature = 380 + 273.15

precipitate_phase = "FCC_L12#2"
matrix_phase = "DIS_FCC_A1"


# static helper method to get the results for the alloys
def get_results(results):
    time_1, mean_radius_1 = results.get_mean_radius_of(precipitate_phase)
    for i in time_1:
        radius, size_distribution = results.get_size_distribution_for_radius_of(precipitate_phase, i)
        for j in range(0, len(size_distribution)):
            if size_distribution[j] < 1.0: size_distribution[j] = float('NaN')
    return time_1, mean_radius_1, radius, size_distribution


with TCPython():
    system = (SetUp()
              .set_cache_folder(os.path.basename(__file__) + "_cache")
              .select_thermodynamic_and_kinetic_databases_with_elements("NIDEMO", "MNIDEMO", ["Ni", "Al", "Cr"])
              .select_phase(matrix_phase)
              .select_phase(precipitate_phase)
              .get_system())

    setup = (system.with_non_isothermal_precipitation_calculation()
             .set_composition_unit(CompositionUnit.MOLE_PERCENT)
             .set_composition("Al", 8.0)
             .set_composition("Cr", 8.0)
             .with_numerical_parameters(NumericalParameters().
                                        set_pre_processing_option(PreProcessingOption.ON_WITH_INCIPIENT_MELTING))
             .with_temperature_profile(TemperatureProfile()
                                       .add_time_temperature(0, initial_temperature)
                                       .add_time_temperature(simulation_time, lower_temperature))
             .set_simulation_time(simulation_time)
             .with_matrix_phase((MatrixPhase(matrix_phase)
                                 .set_mobility_adjustment("all", 5.0, 0.0)
                                 .add_precipitate_phase((PrecipitatePhase(precipitate_phase)
                                                         .set_interfacial_energy(0.023)
                                                         )))))

    result_8 = setup.calculate()

    result_10 = setup.set_composition("Al", 10.0).set_composition("Cr", 10.0).calculate()

    # use a static method to get the results for both alloys
    t8, m8, r8, s8 = get_results(result_8)
    t10, m10, r10, s10 = get_results(result_10)

    # mean radius predicted for Ni8Al-8Cr alloy
    fig1, ax1 = plt.subplots()
    ax1.semilogy(t8, m8, "g-")
    ax1.set_title(r"Mean radius of FCC_L12#2 precipitate for Ni8Al-8Cr (at%)", fontsize=13)
    ax1.set_xlabel("Time [s]")
    ax1.set_ylabel("Length [m]")
    ax1.legend(["Mean radius of FCC_L12#2"], loc=1)
    plt.tight_layout()
    plt.show()

    # mean radius predicted for Ni10Al-10Cr alloy
    fig2, ax2 = plt.subplots()
    ax2.semilogy(t10, m10, "g-")
    ax2.set_title(r"Mean radius of FCC_L12#2 precipitate for Ni10Al-10Cr (at%)", fontsize=13)
    ax2.set_xlabel("Time [s]")
    ax2.set_ylabel("Length [m]")
    ax2.legend(["Mean radius of FCC_L12#2"], loc=1)
    plt.tight_layout()
    plt.show()

    # size distribution at the end of heat treatment for Ni8Al-8Cr alloy
    fig4, ax4 = plt.subplots()
    ax4.set_title(r"Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax4.set_xlabel("Length [m]")
    ax4.set_ylabel("Size distribution [m$^{-4}$]")
    ax4.loglog(r8, s8)
    ax4.legend(["Size distribution - Ni8Al-8Cr alloy"], loc=1)
    plt.tight_layout()
    plt.show()

    # size distribution at the end of heat treatment for Ni10Al-10Cr alloy
    fig5, ax5 = plt.subplots()
    ax5.set_title(r"Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax5.set_xlabel("Length [m]")
    ax5.set_ylabel("Size distribution [m$^{-4}$]")
    ax5.loglog(r10, s10)
    plt.legend(["Size distribution - Ni10Al-10Cr alloy"], loc=1)
    plt.tight_layout()
    plt.show()

    # 3D size distributions for alloys Ni8Al-8Cr and Ni10Al-10Cr
    fig6, ax6 = plt.subplots()
    ax6.set_title(r"Comparing Ni8Al-8Cr and Ni10Al-10Cr size distributions", fontsize=13)
    ax6.set_xlabel("Length [m]")
    ax6.set_ylabel("Size distribution [m$^{-4}$]")
    ax6.loglog(r8, s8, "r-", r10, s10, "b-.")
    ax6.legend(["Ni8Al-8Cr", "Ni10Al-10Cr"], loc=1)
    plt.tight_layout()
    plt.show()

    # 3D mean size population data for Ni10Al-10Cr
    is_sectioned = False
    split_3d_rbar_data = result_10.get_split_mean_radius_of(precipitate_phase, is_sectioned)

    fig7, ax7 = plt.subplots()
    i = 0
    for rbar_data in split_3d_rbar_data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        rbar = []
        for datapoint in rbar_data:
            time.append(datapoint[0])
            rbar.append(datapoint[1])
            if (rbar[-1] < 5E-10):
                rbar[-1] = float('NaN')
        ax7.plot(time, rbar, label=lengendStr)
    ax7.set_title(r"Mean 3D radius of FCC_L12#2 precipitate populations", fontsize=13)
    ax7.set_xlabel("time [s]")
    ax7.set_ylabel("Radius [m]")
    ax7.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 2D mean size population data for Ni10Al-10Cr
    is_sectioned = True
    split_2d_rbar_Data = result_10.get_split_mean_radius_of(precipitate_phase, is_sectioned)

    fig8, ax8 = plt.subplots()
    i = 0
    for rbar_data in split_2d_rbar_Data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        rbar = []
        for datapoint in rbar_data:
            time.append(datapoint[0])
            rbar.append(datapoint[1])
            if (rbar[-1] < 5E-10):
                rbar[-1] = float('NaN')
        ax8.plot(time, rbar, label=lengendStr)
    ax8.set_title(r"Mean 2D radius of FCC_L12#2 precipitate populations", fontsize=13)
    ax8.set_xlabel("time [s]")
    ax8.set_ylabel("Radius [m]")
    ax8.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 3D number density population data for Ni10Al-10Cr
    is_sectioned = False
    split_3d_nv_data = result_10.get_split_number_density_of(precipitate_phase, is_sectioned)

    fig9, ax9 = plt.subplots()
    i = 0
    for nv_data in split_3d_nv_data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        nv = []
        for datapoint in nv_data:
            time.append(datapoint[0])
            nv.append(datapoint[1])
            if (nv[-1] < 1):
                nv[-1] = float('NaN')
        ax9.semilogy(time, nv, label=lengendStr)
    ax9.set_title(r"3D number density of FCC_L12#2 precipitate populations", fontsize=13)
    ax9.set_xlabel("time [s]")
    ax9.set_ylabel("number density [m$^{-3}$]")
    ax9.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 2D number density population data for Ni10Al-10Cr
    is_sectioned = True
    split_3d_nv_data = result_10.get_split_number_density_of(precipitate_phase, is_sectioned)

    fig10, ax10 = plt.subplots()
    i = 0
    for nv_data in split_3d_nv_data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        nv = []
        for datapoint in nv_data:
            time.append(datapoint[0])
            nv.append(datapoint[1])
            if (nv[-1] < 1):
                nv[-1] = float('NaN')
        ax10.semilogy(time, nv, label=lengendStr)
    ax10.set_title(r"2D number density of FCC_L12#2 precipitate populations", fontsize=13)
    ax10.set_xlabel("time [s]")
    ax10.set_ylabel("number density [m$^{-2}$]")
    ax10.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 3D volume fraction population data for Ni10Al-10Cr
    is_sectioned = False
    split_3d_vf_data = result_10.get_split_volume_or_area_fraction_of(precipitate_phase, is_sectioned)

    fig11, ax11 = plt.subplots()
    i = 0
    for vf_data in split_3d_vf_data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        vf = []
        for datapoint in vf_data:
            time.append(datapoint[0])
            vf.append(datapoint[1])
            if (vf[-1] < 1E-10):
                vf[-1] = float('NaN')
        ax11.plot(time, vf, label=lengendStr)
    ax11.set_title(r"3D volume fraction of FCC_L12#2 precipitate populations", fontsize=13)
    ax11.set_xlabel("time [s]")
    ax11.set_ylabel("volume fraction")
    ax11.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 2D area fraction population data for Ni10Al-10Cr
    is_sectioned = True
    split_2d_af_data = result_10.get_split_volume_or_area_fraction_of(precipitate_phase, is_sectioned)

    fig12, ax12 = plt.subplots()
    i = 0
    for af_data in split_2d_af_data:
        i += 1
        lengendStr = 'population#' + str(i)
        time = []
        af = []
        for datapoint in af_data:
            time.append(datapoint[0])
            af.append(datapoint[1])
            if (af[-1] < 1E-10):
                af[-1] = float('NaN')
        ax12.plot(time, af, label=lengendStr)
    ax12.set_title(r"2D area fraction of FCC_L12#2 precipitate populations", fontsize=13)
    ax12.set_xlabel("time [s]")
    ax12.set_ylabel("area fraction")
    ax12.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 3D size distribution for Ni10Al-10Cr
    is_sectioned = False
    split_distributions = result_10.get_split_size_distribution_for_radius_of(precipitate_phase, simulation_time,
                                                                              is_sectioned)

    fig13, ax13 = plt.subplots()
    i = 0
    for disp in split_distributions:
        i += 1
        lengendStr = 'population#' + str(i)
        r = []
        f = []
        for datapoint in disp:
            r.append(datapoint[0])
            f.append(datapoint[1])
        ax13.loglog(r, f, label=lengendStr)
    ax13.set_title(r"3D Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax13.set_xlabel("Length [m]")
    ax13.set_ylabel("Size distribution [m$^{-4}$]")
    ax13.legend(loc="best")
    plt.tight_layout()
    plt.show()

    # 2D size distribution for Ni10Al-10Cr
    is_sectioned = True
    split_2d_sistributions = result_10.get_split_size_distribution_for_radius_of(precipitate_phase, simulation_time,
                                                                                 is_sectioned)

    fig14, ax14 = plt.subplots()
    i = 0
    for disp in split_2d_sistributions:
        i += 1
        lengendStr = 'population#' + str(i)
        r = []
        f = []
        for datapoint in disp:
            r.append(datapoint[0])
            f.append(datapoint[1])
        ax14.loglog(r, f, label=lengendStr)
    ax14.set_title(r"2D Size distribution of FCC_L12#2 precipitate", fontsize=13)
    ax14.set_xlabel("Length [m]")
    ax14.set_ylabel("Size distribution [m$^{-3}$]")
    ax14.legend(loc="best")
    plt.tight_layout()
    plt.show()
