from tc_python import *
from multiprocessing import Manager, cpu_count
import concurrent.futures
import time

"""
Parallel Processing for Multiple Additive Manufacturing Simulations using TC-Python

This example demonstrates how to efficiently run multiple additive manufacturing (AM) simulations 
in parallel using TC-Python with Python's multiprocessing capabilities. The code implements a 
systematic approach to handle multiple simulations with different process parameters (power and 
scanning speed).

The simulation setup:
- Uses a Gaussian heat source model with calculated absorptivity
- Material: Inconel 625 (IN625)
- Process parameters are varied:
  * Scanning speeds: 100-350 mm/s
  * Laser powers: 100-350 W
- Each simulation runs on a specified number of cores
- Results include melt pool dimensions (width, depth, and length)
"""

def run_single_simulation(params):
    """
    Run a single AM simulation with the given parameters
    """
    index, cores, speed, power, available_cores, lock = params
    while True:
        with lock:
            if available_cores.value >= cores:
                available_cores.value -= cores
                break
        time.sleep(1)  # Wait before retry

    try:
        print("Starting task {:2d} with {:2d} cores. Remaining: {:2d} cores. Power={:3d} W, Speed={:.2f} m/s".
              format(index, cores, available_cores.value, power, speed))

        with TCPython() as start:
            start.set_cache_folder("cache")

            mp = MaterialProperties.from_library("IN625")

            calc = (start.with_additive_manufacturing()
                    .with_steady_state_calculation()
                    .with_numerical_options(NumericalOptions().set_number_of_cores(cores))
                    .with_material_properties(mp)
                    .disable_fluid_flow_marangoni()
                    .with_heat_source(HeatSource.gaussian_with_calculated_absorptivity()
                                      .set_power(power)
                                      .set_scanning_speed(speed)))

            result = calc.calculate()

            # Extract the data we need from the result object before returning
            data = {
                'power': power,
                'scanning_speed': speed,
                'meltpool_width': result.get_meltpool_width(),
                'meltpool_depth': result.get_meltpool_depth(),
                'meltpool_length': result.get_meltpool_length()
            }

        return data
    finally:
        with lock:
            available_cores.value += cores
            print(f"Completed task {index}. Released {cores} cores. Now available: {available_cores.value}")

if __name__ == '__main__':
    # Configuration
    max_workers = 6
    cores_per_task = 4
    speed = [100e-3, 150e-3, 200e-3, 250e-3, 300e-3, 350e-3]
    power = [100, 150, 200, 250, 300, 350]
    results = []
    total_cores = cpu_count()

    # Create a manager to share objects between processes
    with Manager() as manager:
        available_cores = manager.Value('i', total_cores)
        lock = manager.Lock()

        print(f"Total CPU cores available: {total_cores}")

        # Prepare parameters for each simulation
        simulation_params = [
            (i, cores_per_task, speed[i], power[i], available_cores, lock) for i in range(len(speed))
        ]

        print(f"Starting {len(simulation_params)} simulations with {max_workers} workers")

        # Run simulations using ProcessPoolExecutor
        with concurrent.futures.ProcessPoolExecutor(max_workers=max_workers) as executor:
            futures = [executor.submit(run_single_simulation, params) for params in simulation_params]

            # Collect results as they complete
            for future in concurrent.futures.as_completed(futures):
                try:
                    result = future.result()
                    if result is not None:
                        results.append(result)
                except Exception as e:
                    print(f"Task failed with error: {e}")

        print(f"Successfully completed {len(results)} tasks on {max_workers} workers. Each worker has {cores_per_task} cores.")

    # Print results
    for result in results:
        try:
            print("Power: {:3d} W, Speed: {:.2f} m/s, Melt pool width:{:.5E} m, depth:{:.5E} m, length:{:.5E} m".
                  format(result['power'], result['scanning_speed'], result['meltpool_width'], result['meltpool_depth'], result['meltpool_length']))
        except Exception as e:
            print(f"Error printing result: {e}")