from tc_python import *
import matplotlib.pyplot as plt
import numpy as np
import pyvista as pv

"""
This example demonstrates how the laser strategy can be visualized before performing an AM calculation to verify the
model setup and obtain useful information for placing probes.

Plots are created for each layer presenting the laser strategy, and the deposition time of the layer.

A line-scan of probes is also created centered in the middle of the representative volume element. If a single point is 
requested, the probe is based upon the a location in the center where the laser passes. For greater number
of probes, the line scan positions the probes uniformly between two mid-points between the center of the domain.
"""

# Laser parameters
laser_power = 230.0
laser_absorptivity = 88.06
laser_scan_speed = 960.0e-3
laser_ar = 97.3e-6
laser_af = 99.04e-6
laser_b = 41.67e-6
laser_c = 181.11e-6

# Laser strategy parameters
num_layers = 3
hatch_spacing = 0.2E-3
angle_between_layers = 30.0

# build parameters
layer_thickness = 55.0e-6
powder_fill_time = 10.0

# domain settings
height = 0.95e-3
width = 2.0e-3
length = 2.0e-3
margin = 0.25e-3

centre = np.zeros(2)
centre[0] = length * 0.5
centre[1] = width * 0.5

# component geometry and lift time calculation
approx_component_geometry = 1E-2
approximate_component_geometry = (approx_component_geometry-max([length, width]))/2
lift_time = approximate_component_geometry/laser_scan_speed

num_probes_per_layer = 4


def line_scan(start_point, end_point, num_points):
    n = len(start_point)
    points = np.zeros([num_points, n], dtype=float)

    if num_points == 1:
        for i in range(0, n):
            points[0][i] = 0.5*(end_point[i] + start_point[i])
    else:
        points[0, 0:n] = start_point

        dx = np.zeros(n, dtype=float)
        for i in range(0, n):
            dx[i] = (end_point[i]-start_point[i])/(float(num_points-1))

        for i in range(1, num_points):
            for j in range(0, n):
                points[i, j] = points[i-1, j] + dx[j]

    return points

with TCPython() as start:
    calc = (start.with_additive_manufacturing().with_transient_calculation()
            .set_height(height)
            .set_width(width)
            .set_length(length)
            .set_layer_thickness(layer_thickness)
            .with_scanning_strategy(ScanningStrategy().bi_directional().
                                    set_hatch_spacing(hatch_spacing).
                                    set_lift_time(lift_time).
                                    set_margin(margin).
                                    set_number_of_layers(num_layers).
                                    set_powder_fill_time(powder_fill_time).
                                    set_angle(angle_between_layers)))

    scanning_path = calc.get_scanning_path()


    for layerID in range(0, num_layers):
        plt.figure()



        scanning_path_positions = scanning_path.get_scanning_path_of_layer(layerID)
        linesPerLayer = len(scanning_path_positions)

        start_of_deposition, t_end = scanning_path.get_time_on_path(0, layerID)
        t_start, end_of_deposition = scanning_path.get_time_on_path(linesPerLayer-1, layerID)

        simulated_deposition_time = end_of_deposition - start_of_deposition
        dep_time_str = "%.4f" % simulated_deposition_time

        figureTitle = "layer#"+str(layerID+1)+", simulated deposition time = "+dep_time_str+"s"
        print(figureTitle)

        for pathID in range(0, linesPerLayer):
            start_Points = scanning_path_positions[pathID][0:3]
            end_Points = scanning_path_positions[pathID][3:6]

            x = start_Points[0]*1000
            y = start_Points[1]*1000
            u = end_Points[0]*1000 - x
            v = end_Points[1]*1000 - y

            plt.arrow(x, y, u, v, head_width=0.025, head_length=0.025, fc='k', ec='k')

        # add a point near the centre as a probe
        point_on_path, factor_, dist_, time_ = scanning_path.get_heat_source_position_nearby(layerID, centre[0], centre[1])

        # find the starting place for a line of probes
        start_probe = scanning_path.get_position_between_two_nearest_lines(layerID, centre[0], centre[1])

        # find the last point for a line of probes
        approx_location = np.zeros(3)
        for i in range(0, 3):
            dp = start_probe[i] - point_on_path[i]
            approx_location[i] = point_on_path[i] - dp
        end_probe = scanning_path.get_position_between_two_nearest_lines(layerID, approx_location[0], approx_location[1])

        # plot and create probes
        probe_points = line_scan(start_probe, end_probe, num_probes_per_layer)

        for i in range(0, num_probes_per_layer):
            plt.scatter(probe_points[i, 0]*1000, probe_points[i, 1]*1000, color='red', marker='x', s=50)
            calc.add_probe(ProbeCoordinate(probe_points[i, 0], probe_points[i, 1], probe_points[i, 2]))


        plt.xlim(0, length*1e3)
        plt.ylim(0, width*1e3)

        plt.title(figureTitle)
        plt.axis('equal')
        plt.xlabel('X axis (mm)')
        plt.ylabel('Y axis (mm)')
        plt.show()