from tc_python import *
import numpy as np

"""
In this example a steady state with Gaussian heat source and keyhole model is demonstrated.
The parameters for the Gaussian heat source are computed using optimization in a steady-state case for the given 
process parameters (power and scanning speed) and the melt pool size reported in the paper by Hu et al. [2019Hu].

[2019Hu] Advances in Materials Science and Engineering, 2019, Article ID 9451406. https://doi.org/10.1155/2019/9451406
"""

with TCPython() as start:
    start.set_cache_folder("cache")

    mp = LibraryMaterialProperties("SS316L")

    hs = HeatSource.gaussian_with_calculated_absorptivity().with_keyhole_model(KeyholeModel()).set_beam_radius(24.68e-6)

    calc = (start.with_additive_manufacturing()
            .with_steady_state_calculation()
            .with_numerical_options(NumericalOptions().set_number_of_cores(4))
            .with_material_properties(mp)
            .disable_fluid_flow_marangoni()
            .with_heat_source(hs)
            .with_mesh(CoarseMesh()))

    results = {}
    P = 100
    for V in [2.8, 1.2, 0.4]:
        hs.set_power(P).set_scanning_speed(V)
        PV = "[{0:3.0f},".format(P) + "{0:4.0f}]".format(V * 1e3)
        results[PV] = calc.calculate()

    for PV, result in results.items():
        if result.has_absorption_distribution_on_keyhole():
            extra_info = ""
        else:
            extra_info = " (On flat surface)"
        print("{}; Has keyhole: {}. Melt pool width:{:.4E}, depth:{:.4E}, length:{:.4E}; Effective absorption under the heat source{}:{:.2f}%"
              .format(PV,
                      result.has_keyhole(),
                      result.get_meltpool_width(),
                      result.get_meltpool_depth(),
                      result.get_meltpool_length(),
                      extra_info,
                      result.get_effective_absorption()*100))

        # plot showing the three different simulation conditions
        plotter, mesh = result.get_pyvista_plotter(shape=(2, 1))
        plotter.add_text("Keyhole", font_size=14)
        plotter.add_mesh(mesh)

        plotter.subplot(1, 0)
        plotter.add_text("Iso-surface solid & liquid", font_size=14)
        contour_mesh = mesh.contour(
            isosurfaces=np.array([mp.get_solidification_temperature(), mp.get_liquidus_temperature()]))
        plotter.add_mesh(mesh.outline(), color="k")
        plotter.add_mesh(contour_mesh)
        abs_grid = result.get_absorptivity_mesh()
        if abs_grid is not None:
            plotter.add_mesh(abs_grid, scalars="absorption", opacity=0.5, color="red", label="Absorptivity Data")

        plotter.link_views()
        plotter.camera_position = 'iso'
        plotter.show()
        del plotter
    
