from tc_python import *
import pyvista as pv
import numpy as np

"""
In this example steady-state simulations are performed for IN625 with different conditions, i.e.
 - without fluid flow in the melt pool, 
 - with fluid flow in the melt pool due to Marangoni effect, 
 - and using separate material properties for the powder. 
 
and then the results are compared to demonstrate the effects of fluid flow and separate material properties for the 
powder on the temperature distribution as well as on the shape of the melt pool.

For the first two simulations, the same material properties are used for both powder and solid substrate while for the
third simulation no fluid flow is included in the melt pool.

The melt-pool area is multiplied by two to account for the symmetry condition used in steady-state simulations.
"""

def get_meltpool_surface_area(mesh, mp):
    mesh_slice_top = mesh.slice(normal='z', origin=(0, 0, 0))
    top_surface = mesh_slice_top.extract_surface()
    mesh_clipped_solid = top_surface.clip_scalar(value=mp.get_liquidus_temperature(), invert=True)
    solid_area = mesh_clipped_solid.area

    top_surface_area = (mesh.bounds[1] - mesh.bounds[0]) * (mesh.bounds[3] - mesh.bounds[2])

    surface_area_melt_pool = top_surface_area - solid_area
    return surface_area_melt_pool

with TCPython() as start:
    start.set_cache_folder("cache")

    mp = MaterialProperties.from_library("IN625")

    calc = (start.with_additive_manufacturing()
            .with_steady_state_calculation()
            .with_numerical_options(NumericalOptions().set_number_of_cores(4))
            .with_material_properties(mp)
            .set_layer_thickness(55.0e-6)
            .disable_fluid_flow_marangoni()
            .with_heat_source(HeatSource.gaussian_with_constant_absorptivity()
                              .set_power(120.0)
                              .set_absorptivity(40.0)
                              .set_beam_radius(100.0e-6)
                              .set_scanning_speed(600.0e-3)))

    result_no_fluid_flow = calc.calculate()
    mp_area_no_fluid_flow = get_meltpool_surface_area(result_no_fluid_flow.get_pyvista_mesh(), mp) * 2

    calc.enable_fluid_flow_marangoni()
    result_fluid_flow = calc.calculate()
    mp_area_fluid_flow = get_meltpool_surface_area(result_fluid_flow.get_pyvista_mesh(), mp) * 2

    calc.enable_separate_materials().set_powder_density(80.0)
    result_powder = calc.calculate()
    mp_area_powder = get_meltpool_surface_area(result_powder.get_pyvista_mesh(), mp) * 2

    print("No fluid flow....................... Melt pool width:{:.4E}m, depth:{:.4E}m, "
          "length:{:.4E}m, surface area:{:.4E}m^2".format(
        result_no_fluid_flow.get_meltpool_width(), result_no_fluid_flow.get_meltpool_depth(),
        result_no_fluid_flow.get_meltpool_length(), mp_area_no_fluid_flow))
    print("Fluid fow........................... Melt pool width:{:.4E}m, depth:{:.4E}m, "
          "length:{:.4E}m, surface area:{:.4E}m^2".format(
        result_fluid_flow.get_meltpool_width(), result_fluid_flow.get_meltpool_depth(),
        result_fluid_flow.get_meltpool_length(), mp_area_fluid_flow))
    print("Separate powder material properties. Melt pool width:{:.4E}m, depth:{:.4E}m, "
          "length:{:.4E}m, surface area:{:.4E}m^2".format(
        result_powder.get_meltpool_width(), result_powder.get_meltpool_depth(),
        result_powder.get_meltpool_length(), mp_area_powder))

    # plot showing the three different simulation conditions
    plotter1, mesh_no_fluid = result_no_fluid_flow.get_pyvista_plotter(shape=(3, 1))
    plotter1.add_text("No fluid flow", font_size=14)
    plotter1.add_mesh(mesh_no_fluid)

    mesh_fluid = result_fluid_flow.get_pyvista_mesh()
    plotter1.subplot(1, 0)
    plotter1.add_text("Fluid flow", font_size=14)
    actor_fluid = plotter1.add_mesh(mesh_fluid)

    mesh_powder = result_powder.get_pyvista_mesh()
    plotter1.subplot(2, 0)
    plotter1.add_text("Separate powder material properties", font_size=14)
    actor_powder = plotter1.add_mesh(mesh_powder)
    plotter1.link_views()
    plotter1.camera_position = 'iso'
    plotter1.show(interactive_update=True)

    # plot showing filtering of powder and liquid
    plotter2, mesh_powder_show_all = result_powder.get_pyvista_plotter(shape=(3, 1))
    plotter2.add_mesh(mesh_powder_show_all)

    mesh_filter_powder = result_powder.get_pyvista_mesh(material_type=MaterialType.POWDER)
    plotter2.subplot(1, 0)
    plotter2.add_text("Filter to show powder only", font_size=14)
    plotter2.add_mesh(mesh_filter_powder)

    mesh_filter_liquid = result_powder.get_pyvista_mesh(material_type=MaterialType.LIQUID)
    plotter2.subplot(2, 0)
    plotter2.add_text("Filter to show liquid only", font_size=14)
    plotter2.add_mesh(mesh_filter_liquid)
    plotter2.link_views()
    plotter2.camera_position = 'iso'
    plotter2.show(interactive_update=True)

    # plot showing flow field, solid & liquid iso-surfaces
    plotter3, mesh_flow = result_fluid_flow.get_pyvista_plotter(shape=(3, 1))
    plotter3.add_mesh(mesh_flow)

    plotter3.subplot(1, 0)
    plotter3.add_text("Iso-surface solid & liquid", font_size=14)
    contour_mesh = mesh_flow.contour(
        isosurfaces=np.array([mp.get_solidification_temperature(), mp.get_liquidus_temperature()]))
    plotter3.add_mesh(mesh_flow.outline(), color="k")
    plotter3.add_mesh(contour_mesh)

    plotter3.subplot(2, 0)
    plotter3.add_text("Flow field", font_size=14)
    scale_flow_field_vectors = 1e-5
    plotter3.add_arrows(mesh_flow.points, mesh_flow.active_vectors, mag=scale_flow_field_vectors,
                       scalar_bar_args={"title": "Velocity*{}".format(scale_flow_field_vectors)})
    plotter3.add_mesh(mesh_flow.outline(), color="k")
    plotter3.link_views()
    plotter3.camera_position = 'iso'
    plotter3.show()

    # plot-over-line
    plotter4, mesh_fluid = result_fluid_flow.get_pyvista_plotter()
    a = [mesh_fluid.bounds[1], mesh_fluid.bounds[2], mesh_fluid.bounds[5]]
    b = [mesh_fluid.bounds[0], mesh_fluid.bounds[2], mesh_fluid.bounds[5]]

    line = pv.Line(a, b)
    plotter4.add_mesh(mesh_fluid.outline(), color="k")
    plotter4.add_mesh(line, color="b", line_width=5)
    mesh_fluid.plot_over_line(a, b, resolution=100, ylabel="Temperature [K]", title="Plot-over-line")
    plotter4.show()
