from tc_python import *
import matplotlib.pyplot as plt

"""
In this example a steady state and a single track simulation (with heat source from steady-state) are performed for 
Inconel 738LC.

The Transient with heat source from Steady-state model exploits the assumption that the melt pool size and temperature
distribution reaches a steady-state almost instantly and thus first solves for a Steady-state case with the given 
process parameters. The result from the Steady-state in the form of temperature distribution in the melt pool is then 
mapped as a heat source in the transient simulation. This novel approach is much faster and efficient than the approach
used in example AM_01 where the heat equation is solved in the entire 3D domain.
"""

with TCPython() as start:
    start.set_cache_folder("cache")
    mp = MaterialProperties.from_library("IN738LC")

    heat_source = (HeatSource.double_ellipsoidal()
                   .set_power(230.0)
                   .set_absorptivity(88.06)
                   .set_ar(97.3e-6)
                   .set_af(99.04e-6)
                   .set_b(41.67e-6)
                   .set_c(181.11e-6)
                   .set_scanning_speed(960.0e-3))

    probe_1 = ProbeCoordinate(0.5e-3, 0.25e-3, 1.005e-3)
    probe_2 = ProbeCoordinate(0.5e-3, 0.29e-3, 1.005e-3)

    calc_transient_ss = (start.with_additive_manufacturing()
                         .with_transient_with_steady_state_calculation()
                         .with_numerical_options(NumericalOptions().set_number_of_cores(4))
                         .with_material_properties(mp)
                         .set_height(0.95e-3).set_width(0.5e-3).set_length(2.0e-3)
                         .set_layer_thickness(55e-6)
                         .with_heat_source(heat_source)
                         .disable_fluid_flow_marangoni()
                         .with_scanning_strategy(ScanningStrategy().single_track().set_margin(0.25e-3))
                         .add_probe(probe_1)
                         .add_probe(probe_2))

    result_transient_ss = calc_transient_ss.calculate()

    probe_1_time, probe_1_temperature = result_transient_ss.get_temperatures_at_probe(probe_1)
    probe_2_time, probe_2_temperature = result_transient_ss.get_temperatures_at_probe(probe_2)

    fig2, axs = plt.subplots(1, 1)
    axs.plot(probe_1_time, probe_1_temperature, marker=".", label="Probe 1")
    axs.plot(probe_2_time, probe_2_temperature, marker=".", label="Probe 2")
    axs.set_xlabel("Time [s]")
    axs.set_ylabel("Temperature [K]")
    plt.legend(loc="best")
    plt.show(block=False)

    # plot update function that will be called every time the slider with simulation time is dragged in the plot windows
    def update_plot(plotter, mesh):
        mesh.clip(normal="y", inplace=True, invert=False)
        plotter.add_mesh(mesh, clim=[296.15, 1.2 * mp.get_evaporation_temperature()], name="mymesh")


    plotter, mesh = result_transient_ss.get_pyvista_plotter(update_plot_callback=update_plot)
    update_plot(plotter, mesh)
    plotter.camera_position = 'iso'
    plotter.show()